/**
  ******************************************************************************
  * @file    cp_priqueue.c 
  * @author  Ruediger R. Asche
  * @version V1.0.0
  * @date    July 14, 2016
  * @brief   implements the priority queue data structure
  ******************************************************************************
  * @attention
  *
  * THE PRESENT FIRMWARE WHICH IS FOR GUIDANCE ONLY AIMS AT PROVIDING CUSTOMERS
  * WITH CODING INFORMATION REGARDING THEIR PRODUCTS IN ORDER FOR THEM TO SAVE
  * TIME. AS A RESULT, THE AUTHOR SHALL NOT BE HELD LIABLE FOR ANY
  * DIRECT, INDIRECT OR CONSEQUENTIAL DAMAGES WITH RESPECT TO ANY CLAIMS ARISING
  * FROM THE CONTENT OF SUCH FIRMWARE AND/OR THE USE MADE BY CUSTOMERS OF THE
  * CODING INFORMATION CONTAINED HEREIN IN CONNECTION WITH THEIR PRODUCTS.
  ******************************************************************************  
  */ 

#include "project.h"

#include "cp.h"

const UBaseType_t g_PriQueueSizes[CP_PRIQUEUE_MAXPRIORITIES] = 
{
    20,
    10,
    3
};

/** @brief Creates a priority queue.
 *
 * 
 *  @param p_QueueHandle (output parameter) receives a handle to the created object
 *
 *  @return status code (see CP_STATUSCODE_XXX enumerator in cp_frame.h)
 *
 * A priority queue is a bundle of FreeRTOS queues which consists of several (in this case 3) priority levels
 * (see enumerator CP_PRILEVELS). A peek on the queue will return the element with the highest priority.
 *
 */

CP_STATUSCODE CP_CreatePriQueue(CP_PRIQUEUEHANDLE *p_QueueHandle)
{
    CP_PRILEVELS a_Loop;
    CP_STATUSCODE a_Return = CP_STATUSCODE_NOMEM;
    *p_QueueHandle = 0;
    
    // first, allocate memory for the queue object
    *p_QueueHandle = (CP_PRIQUEUEHANDLE) pvPortMalloc(sizeof(CP_PRIQUEUE));
    if (*p_QueueHandle)
    {
        memset(*p_QueueHandle,0,sizeof(CP_PRIQUEUE));
        // then create all queues.
        for (a_Loop = CP_PRIQUEUE_CRITICALPRI; a_Loop < CP_PRIQUEUE_MAXPRIORITIES; a_Loop++)
        {
            xQueueHandle aNextItem = xQueueCreate(g_PriQueueSizes[a_Loop],sizeof(struct CP_MSGPUMPINFO));
            if (!aNextItem)
            {
                CP_DeletePriQueue(*p_QueueHandle);
                return a_Return; 
            }
            else
                (*p_QueueHandle)->m_Queues[a_Loop] = aNextItem;            
        }
        (*p_QueueHandle)->m_Mutex = xSemaphoreCreateRecursiveMutex();
        if (!(*p_QueueHandle)->m_Mutex)
        {
            CP_DeletePriQueue(*p_QueueHandle);
            return a_Return; 
        }
        (*p_QueueHandle)->m_QueueCurrentlySending = CP_NOQUEUE;
        a_Return = CP_STATUSCODE_SUCCESS;
    }
    return a_Return;
}

/** @brief Deletes a priority queue.
 *
 * 
 *  @param p_QueueHandle (output parameter) receives a handle to the created object
 *
 *  @return status code (see CP_STATUSCODE_XXX enumerator in cp_frame.h)
 *
 */

CP_STATUSCODE CP_DeletePriQueue(CP_PRIQUEUEHANDLE p_QueueHandle)
{
    if (p_QueueHandle)
    {
        CP_PRILEVELS a_Loop;
        for (a_Loop = CP_PRIQUEUE_CRITICALPRI; a_Loop < CP_PRIQUEUE_MAXPRIORITIES; a_Loop++)
        {
            if (p_QueueHandle->m_Queues[a_Loop])
                vQueueDelete(p_QueueHandle->m_Queues[a_Loop]);
        }
        if (p_QueueHandle->m_Mutex)
            vSemaphoreDelete(p_QueueHandle->m_Mutex);
        vPortFree(p_QueueHandle);
    }
    return CP_STATUSCODE_SUCCESS;
}

#define GRACEPERIOD 200

/** @brief Adds an element to a priority queue on a given level.
 *
 * 
 *  @param p_QueueHandle handle to a priority object
 *  @param p_QueueIndex priority level to insert the queue into
 *  @param p_ItemToQueue Data to put into queue. Must be consistent in size with the creation of the queue.
 *
 *  @return status code (see CP_STATUSCODE_XXX enumerator in cp_frame.h)
 *
 * Attention: In this implementation, there is a deadlock potential. This function will back off continously until someone has
 * removed an item from the queue. As a consequence, the task in whose context this is called may be suspended indefinetely, and in the
 * worst case (if it is the task running the message pump which is the only task to remove items from the queue) deadlock itself -
 * not because of the atomic mutex (it is recursive) but because of the queue.
 * In version 1.0, this will not happen because the message pump task will only post to the medium and high priority levels, which are
 * of sufficent size, and applications only post to the lowest level which means application tasks can block indefinitely (eg if the communication stalls).
 * Normally one would implement a high level timeout that will the flush the queue at defined times; however that implies data loss.
 */

CP_STATUSCODE CP_SendPriQueue(CP_PRIQUEUEHANDLE p_QueueHandle,CP_PRILEVELS p_QueueIndex,void *p_ItemToQueue)
{
    CP_STATUSCODE a_Return = CP_STATUSCODE_CORRUPT;
    CP_BeginAtomic(p_QueueHandle);    
    do
    {
        a_Return = (xQueueSend(p_QueueHandle->m_Queues[p_QueueIndex],p_ItemToQueue,GRACEPERIOD) == pdTRUE)?CP_STATUSCODE_SUCCESS:CP_STATUSCODE_CORRUPT;
        if (a_Return != CP_STATUSCODE_SUCCESS)
        {
            CP_EndAtomic(p_QueueHandle);
            vTaskDelay(2);
            CP_BeginAtomic(p_QueueHandle);
        } 
    } while (a_Return != CP_STATUSCODE_SUCCESS);
    CP_EndAtomic(p_QueueHandle);
    return a_Return;     
}

/** @brief Retrieves an element from a priority queue without removing the element.
 *
 * 
 *  @param p_QueueHandle handle to a priority object
 *  @param p_ItemToPeek Storage to receive the retrieved data from queue. Must be consistent in size with the creation of the queue.
 *  @param p_Pri priority level requested. If CP_NOQUEUE is passed, the highest priority element will be retrieved.
 *
 *  @return status code (see CP_STATUSCODE_XXX enumerator in cp_frame.h)
 *
 */

CP_STATUSCODE CP_PeekPriQueue(CP_PRIQUEUEHANDLE p_QueueHandle,void *p_ItemToPeek,CP_PRILEVELS *p_Pri)
{
    CP_PRILEVELS a_Loop;
    if (*p_Pri > CP_NOQUEUE)
        return ((xQueuePeek(p_QueueHandle->m_Queues[a_Loop],p_ItemToPeek,0) == pdTRUE)?CP_STATUSCODE_SUCCESS:CP_STATUSCODE_NODATA);
    for (a_Loop = CP_PRIQUEUE_CRITICALPRI; a_Loop < CP_PRIQUEUE_MAXPRIORITIES; a_Loop++)
    {
        if (xQueuePeek(p_QueueHandle->m_Queues[a_Loop],p_ItemToPeek,0) == pdTRUE)
        {
            *p_Pri = a_Loop;
            return CP_STATUSCODE_SUCCESS;
        };
    }
    return CP_STATUSCODE_NODATA;
}

/** @brief Removes an element from a priority queue.
 *
 * 
 *  @param p_QueueHandle handle to a priority object
 *  @param p_Pri priority level to remove the element from .
 *
 *  @return status code (see CP_STATUSCODE_XXX enumerator in cp_frame.h)
 *
 */

CP_STATUSCODE CP_RemovedPeekedPriQueue(CP_PRIQUEUEHANDLE p_QueueHandle,CP_PRILEVELS p_Pri)
{
    CP_MSGPUMPINFO a_Discard;
    if (p_Pri == CP_NOQUEUE) STATIC_SANITY_CHECK;
    xQueueHandle a_Queue = p_QueueHandle->m_Queues[p_Pri];
    if (!a_Queue)
        STATIC_SANITY_CHECK; 
    if (xQueueReceive(a_Queue,&a_Discard,portMAX_DELAY) == pdTRUE)
        return CP_STATUSCODE_SUCCESS;
    return CP_STATUSCODE_NODATA;  // can't really be unless the reading access to the queue is unserialized which is an error
}

/** @brief Requests exclusive access to a priority queue.
 *
 * 
 *  @param p_QueueHandle handle to a priority object
 *
 *  @return none
 *
 * This function is provided for compound accesses to the queue. Sometimes it must be ensured that several operations on a queue
 * do not get interrupted, for example if an element is temporarily removed and then rescheduled modified. In order to prevent
 * race conditions that leave the queue inconsistent, the caller can use this high level mutual exclusion.
 */

void CP_BeginAtomic(CP_PRIQUEUEHANDLE p_QueueHandle)
{
    xSemaphoreTakeRecursive(p_QueueHandle->m_Mutex,portMAX_DELAY);
}

/** @brief Releases exclusive access to a priority queue.
 *
 * 
 *  @param p_QueueHandle handle to a priority object
 *
 *  @return none
 */

void CP_EndAtomic(CP_PRIQUEUEHANDLE p_QueueHandle)
{
    xSemaphoreGiveRecursive(p_QueueHandle->m_Mutex);
}



